package MusicLandscape.entities;

/**
 * represents the appearance of an artist in (a certain edition of) a TVShow as a specific event. e.g.
 * "Johnny Cash" on "The Tonight Show".
 *
 * @author Jonas Altrock (ew20b126@technikum-wien.at)
 * @version 1
 * @since ExerciseSheet03
 */
public class TVShow extends Event {
    /**
     * the name of this TV show
     * <p>
     * The name of a TVShow cannot be empty or composed of whitespace only.
     */
    private String name;

    /**
     * the number of viewers of this TVShow
     * <p>
     * must be non-negative, 0 means unknown
     */
    private int viewers = 0;

    /**
     * creates a default TVShow
     * <p>
     * a default TVShow is a default event with an unknown name and an unknown number of viewers.
     */
    public TVShow() {
        super();
    }

    /**
     * creates a TV show from an event
     * <p>
     * this constructor performs some kind of promotion such that it takes a generic event and creates
     * a TV show which is a (deep) copy of the original event. The resulting TV show has unknown name and
     * unknown viewers.
     *
     * @param e - the event to copy/promote to TV show
     */
    public TVShow(Event e) {
        super(e);
    }

    /**
     * creates a deep copy of a TVShow
     *
     * @param tvs - the TV show to copy
     */
    public TVShow(TVShow tvs) {
        super(tvs);

        name = tvs.name;
        viewers = tvs.viewers;
    }

    /**
     * gets the number of viewers of this TVShow
     *
     * @return the viewers
     */
    public int getViewers() {
        return viewers;
    }

    /**
     * sets the viewers of this TVshow
     * <p>
     * illegal arguments are ignored
     *
     * @param v - the number of viewers to set
     */
    public void setViewers(int v) {
        if (v < 0) {
            return;
        }
        viewers = v;
    }

    /**
     * gets the name of this TVShow
     *
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * sets the name of this TVShow
     * <p>
     * illegal arguments are ignored
     *
     * @param name - the name to set
     */
    public void setName(String name) {
        if (name == null || name.isBlank()) {
            return;
        }

        this.name = name;
    }

    /**
     * returns a String representation of this TV
     * <p>
     * show the string representation of a TV show is (without quotes):
     * <pre>
     * "artist name" @ "show name" on "date"
     * "description"
     * ("audience" attending ("impact"))
     * </pre>
     * audience are all viewers and attendees of a show
     *
     * @return the string representation
     */
    public String toString() {
        return getArtist().getName() + " @ " + name + " on " + getDate().toString() + "\n" +
                getDescription() + "\n" +
                "(" + (viewers + getAttendees()) + " attending (" + impact() + "))";
    }

    /**
     * returns the impact of this event
     * <p>
     * the impact is an estimation of the number of people who took notice of this event for a TV show
     * event, the impact is the audience times 2. audience are all viewers and attendees of a show
     *
     * @return the impact
     */
    public int impact() {
        return (viewers + getAttendees()) * 2;
    }
}
